<?php
/**
 * Plugin Name: Download Button for Digital Products
 * Plugin URI: https://wordpress.org/plugins/download-button-for-digital-products/
 * Description: Disables the "Add to Cart" button for products a user has already purchased and provides a customizable "Download Now" button for downloadable products.
 * Version: 1.1
 * Author: ZENOR BD
 * Author URI: https://zenorbd.com
 * Text Domain: download-button-for-digital-products
 * Requires at least: 5.0
 * Requires PHP: 7.0
 * WC requires at least: 3.0
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @package Download_Button_For_Digital_Products
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants
define( 'DBDP_VERSION', '1.1' );
define( 'DBDP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'DBDP_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );

// Include settings page
require_once DBDP_PLUGIN_PATH . 'includes/settings-page.php';

// Enqueue plugin styles
function dbdp_enqueue_styles() {
	if ( get_option( 'dbdp_enable_feature', 1 ) == 1 ) {
		wp_enqueue_style( 'dbdp-styles', DBDP_PLUGIN_URL . 'assets/css/plugin-style.css', array(), DBDP_VERSION );
		
		// Apply custom styles
		$button_color = get_option( 'dbdp_download_button_color', '#4CAF50' );
		$hover_color = get_option( 'dbdp_download_button_hover_color', '#45a049' );
		$text_color = get_option( 'dbdp_download_button_text_color', '#fff' );
		$border_radius = get_option( 'dbdp_button_border_radius', '5' ) . 'px';
		$padding = get_option( 'dbdp_button_padding', '12 24' );
		$padding_values = explode(' ', $padding);
		$vertical_padding = isset($padding_values[0]) ? $padding_values[0] . 'px' : '12px';
		$horizontal_padding = isset($padding_values[1]) ? $padding_values[1] . 'px' : '24px';
		
		$custom_css = "
			.dbdp-download-button {
				background-color: {$button_color} !important;
				color: {$text_color} !important;
				border-radius: {$border_radius} !important;
				padding: {$vertical_padding} {$horizontal_padding} !important;
				display: inline-block !important;
				text-decoration: none !important;
				font-weight: 600 !important;
				text-align: center !important;
				transition: all 0.3s ease !important;
				border: none !important;
				cursor: pointer !important;
				margin-top: 15px !important;
			}
			.dbdp-download-button:hover {
				background-color: {$hover_color} !important;
				transform: translateY(-1px) !important;
				box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15) !important;
			}
		";
		
		wp_add_inline_style( 'dbdp-styles', $custom_css );
	}
}
add_action( 'wp_enqueue_scripts', 'dbdp_enqueue_styles' );

// Enqueue admin styles and scripts
function dbdp_enqueue_admin_assets( $hook ) {
	if ( 'settings_page_dbdp-settings' !== $hook ) {
		return;
	}
	
	wp_enqueue_style( 'dbdp-admin-styles', DBDP_PLUGIN_URL . 'assets/css/admin-style.css', array(), DBDP_VERSION );
	wp_enqueue_style( 'wp-color-picker' );
	wp_enqueue_script( 'dbdp-admin-script', DBDP_PLUGIN_URL . 'assets/js/admin-script.js', array( 'jquery', 'wp-color-picker' ), DBDP_VERSION, true );
}
add_action( 'admin_enqueue_scripts', 'dbdp_enqueue_admin_assets' );

// Register settings
function dbdp_register_settings() {
	register_setting( 'dbdp_settings_group', 'dbdp_enable_feature' );
	register_setting( 'dbdp_settings_group', 'dbdp_download_button_text' );
	register_setting( 'dbdp_settings_group', 'dbdp_download_button_color' );
	register_setting( 'dbdp_settings_group', 'dbdp_button_border_radius' );
	register_setting( 'dbdp_settings_group', 'dbdp_download_button_hover_color' );
	register_setting( 'dbdp_settings_group', 'dbdp_download_button_text_color' );
	register_setting( 'dbdp_settings_group', 'dbdp_button_padding' );
	
	// Set default values if they don't exist
	if ( get_option( 'dbdp_enable_feature' ) === false ) {
		update_option( 'dbdp_enable_feature', 1 );
	}
	if ( get_option( 'dbdp_download_button_text' ) === false ) {
		update_option( 'dbdp_download_button_text', 'Download Now' );
	}
	if ( get_option( 'dbdp_download_button_color' ) === false ) {
		update_option( 'dbdp_download_button_color', '#4CAF50' );
	}
	if ( get_option( 'dbdp_button_border_radius' ) === false ) {
		update_option( 'dbdp_button_border_radius', '5' );
	}
	if ( get_option( 'dbdp_download_button_hover_color' ) === false ) {
		update_option( 'dbdp_download_button_hover_color', '#45a049' );
	}
	if ( get_option( 'dbdp_download_button_text_color' ) === false ) {
		update_option( 'dbdp_download_button_text_color', '#fff' );
	}
	if ( get_option( 'dbdp_button_padding' ) === false ) {
		update_option( 'dbdp_button_padding', '12 24' );
	}
}
add_action( 'admin_init', 'dbdp_register_settings' );

// Add settings page in WordPress admin
function dbdp_add_settings_page() {
	add_options_page( 
		'Download Button Settings', 
		'Download Button for Digital Products', 
		'manage_options', 
		'dbdp-settings', 
		'dbdp_render_settings_page' 
	);
}
add_action( 'admin_menu', 'dbdp_add_settings_page' );

// Disable "Add to Cart" for purchased products if feature is enabled
add_filter( 'woocommerce_is_purchasable', 'dbdp_disable_add_to_cart_for_purchased', 10, 2 );
function dbdp_disable_add_to_cart_for_purchased( $purchasable, $product ) {
	if ( get_option( 'dbdp_enable_feature', 1 ) == 1 && is_user_logged_in() ) {
		$user_id = get_current_user_id();
		$cache_key = 'user_bought_product_' . $user_id . '_' . $product->get_id();
		$purchased = get_transient( $cache_key );

		if ( false === $purchased ) {
			$purchased = wc_customer_bought_product( '', $user_id, $product->get_id() );
			set_transient( $cache_key, $purchased, 12 * HOUR_IN_SECONDS );
		}

		if ( $purchased ) {
			return false;
		}
	}
	return $purchasable;
}

// Show "Download Now" link if purchased and feature is enabled
add_action( 'woocommerce_single_product_summary', 'dbdp_show_download_link_if_purchased', 31 );
function dbdp_show_download_link_if_purchased() {
	if ( get_option( 'dbdp_enable_feature', 1 ) == 1 && is_user_logged_in() ) {
		global $product;
		$user_id = get_current_user_id();

		if ( wc_customer_bought_product( '', $user_id, $product->get_id() ) ) {
			$button_text = get_option( 'dbdp_download_button_text', 'Download Now' );
			
			// Get the first available download URL for the user
			$downloads = wc_get_customer_available_downloads( $user_id );
			$download_url = ! empty( $downloads ) ? $downloads[0]['download_url'] : '#';
			
			echo '<a class="dbdp-download-button" href="' . esc_url( $download_url ) . '">' . esc_html( $button_text ) . '</a>';
		}
	}
}

// Add custom "Go to Settings" link and Author's website link to the plugins page
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'dbdp_add_settings_link' );
function dbdp_add_settings_link( $links ) {
	// Add the "Go to Settings" link
	$settings_link = '<a href="' . admin_url( 'options-general.php?page=dbdp-settings' ) . '">' . esc_html__( 'Settings', 'download-button-for-digital-products' ) . '</a>';
	
	// Add the "Author Website" link
	$author_link = '<a href="https://zenorbd.com" target="_blank">' . esc_html__( 'Author Website', 'download-button-for-digital-products' ) . '</a>';

	// Add the new links
	array_unshift( $links, $settings_link );
	$links[] = $author_link;

	return $links;
}

// Load text domain for translations
function dbdp_load_textdomain() {
	load_plugin_textdomain( 'download-button-for-digital-products', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}
add_action( 'plugins_loaded', 'dbdp_load_textdomain' );